
/**
 *  LIFE.CPP
 * 
 * - Quick Description -
 * This program is intended to implement Conway's game of life using:
 * - linear code
 * - a grid and starting layout described by a text input file,
 * OR
 * - a randomly generated grid
 * 
 * It was originally a straight C file but my previous timing code no longer
 * works (something was changed in an update of mingw).  So now it is all C
 * code with a C++ timing function.
 * 
 *  Written by: Brent Feulbach
 *  Date: 20-Aug-2020
 * 
 *  *** _ATTRIBUTION_ ***
 * a)
 * The code in "gif.h" was written by Charlie Tangora.
 * It is public domain code.
 * It was retrieved from: 
 *  https://github.com/charlietangora/gif-h
 *
 * b)
 * The code in the 'measure_time' function is based on the example
 * at: http://www.cplusplus.com/reference/chrono/steady_clock/?kw=steady_clock
 * That said, there is only realy one way to time code - get a time measurement
 * before and after and subtract the 'before' from the 'after'.
 * 
 */


#include "life.h"
// #include "gif.cpp"

// USE_FLOATS_STRING and USE_BITMAP_STRING are used to display which version
//   of the code has been built
#if (USE_FLOATS != 0)
    #define USE_FLOATS_STRING "use floats"
#else
    #define USE_FLOATS_STRING "no floats"
#endif

#if (USE_BITMAP != 0)
    #define USE_BITMAP_STRING "bitmap"
#else
    #define USE_BITMAP_STRING "char map"
#endif

#ifdef DEBUG
    #define VER_STRING "debug"
#else
    #define VER_STRING "release"
#endif



/* GLOBALS */

uint64_t        gridHeight, gridWidth,  // grid dimensions
                numTicks;               // how many generations to run life for

GRID_UNITS      *grids[NUM_GRIDS],      // pointers for the grid buffers
                *curGrid, *nextGrid;    // pointers to the cur and next grid
                                        //   for quick access

char            *fileBuf;               // memory buffer for the input file
uint64_t        fileBufSize,            // size of 'fileBuf' 
                gridBufSize,            // size (in bytes) of each of the grids
                unitsPerRow;             // how many ints represent 1 row in grid

bool            randGrid = false,       // TRUE when 'usage2' is specified
                forceLexParse = false,  // forces input file to be lex parsed
                forceMapParse = false;  // forces input file to be map parsed

char            *gifFn = NULL,          // name of gif output file, if any
                *inFn = NULL;           // name of input (seed) file
uint64_t        gifFrameDelay = DEFAULT_GIF_DELAY;
                                        // how long between gif frames (in MS)

double          initLiveChance = 0.02,  // init chance of a cell being alive
                randLifeChance = 0.0005,// chance per tick that dead cell
                                        //   will come to life
                randDeathChance = 0.0001;// chance per tick that live cell will
                                        //   die
uint64_t        randSeed = 987654321;   // RNG seed

OutputOptions   outOpts;                // options for displaying output

bool            signonPrinted=false;    // Ensure signon is only printed once
bool            printDbg=false;         // Used during debugging to print
                                        //   messages or not

/* 
 * 'correct' is a look-up table for the correct outputs for any 3x3 grid.
 * It was generated by this program using the '-c' command line option.
 * 
 *  a | b | c
 *  - + - + -
 *  d | e | f
 *  - + - + -
 *  g | h | i
 * 
 * The grids are encoded in the lowest 9 bits.  The lowest bit in the encoding
 * is cell 'a' in the diagram above.  The second lowest bit is 'b'.  The third
 * lowest is 'c'.  And so forth.
 * 
 * To look up the correct output for a grid, it should be encoded as described
 * above and the number represented by the encoded grid is the 'look-up' value.
 * 
 * For example, to look up the grid (lets call it "cross")
 *    O.O
 *    .O.
 *    O.O
 * you would encode it as described above.  This would give you the value,
 * 01 0101 0101b = 0x177
 * so 'correct[0x177]' will give the correct values for the next generation of
 * the "cross" grid.
 */
uint16_t correct[] = {
    0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0012, 
    0x0000, 0x0000, 0x0000, 0x001B, 0x0000, 0x0012, 0x0012, 0x000B, 
    0x0000, 0x0000, 0x0000, 0x001B, 0x0000, 0x0012, 0x0036, 0x003F, 
    0x0000, 0x001B, 0x001B, 0x001B, 0x0012, 0x0019, 0x003F, 0x002D, 
    0x0000, 0x0000, 0x0000, 0x0012, 0x0000, 0x0012, 0x0036, 0x0026, 
    0x0000, 0x0012, 0x0012, 0x000B, 0x0012, 0x0000, 0x0026, 0x002D, 
    0x0000, 0x0012, 0x0036, 0x003F, 0x0036, 0x0034, 0x0036, 0x002D, 
    0x0092, 0x0099, 0x00BF, 0x00AD, 0x00B4, 0x00AD, 0x00AD, 0x00AD, 
    0x0000, 0x0000, 0x0000, 0x0018, 0x0000, 0x0010, 0x0010, 0x000A, 
    0x0000, 0x0018, 0x0018, 0x000B, 0x0010, 0x000A, 0x000A, 0x000B, 
    0x0000, 0x0018, 0x0018, 0x0013, 0x0010, 0x001A, 0x003E, 0x0027, 
    0x00D8, 0x00DB, 0x00DB, 0x00C3, 0x00DA, 0x00C9, 0x00EF, 0x00E5, 
    0x0000, 0x0010, 0x0010, 0x000A, 0x0010, 0x0002, 0x0026, 0x002E, 
    0x0090, 0x008A, 0x008A, 0x008B, 0x0082, 0x0088, 0x00AE, 0x00AD, 
    0x0090, 0x009A, 0x00BE, 0x00A7, 0x00B6, 0x00AC, 0x00AE, 0x00A5, 
    0x005A, 0x0049, 0x006F, 0x0065, 0x006C, 0x006D, 0x006D, 0x0065, 
    0x0000, 0x0000, 0x0000, 0x0018, 0x0000, 0x0010, 0x0030, 0x002A, 
    0x0000, 0x0018, 0x0018, 0x000B, 0x0010, 0x000A, 0x002A, 0x002B, 
    0x0000, 0x0018, 0x0038, 0x0033, 0x0030, 0x003A, 0x001E, 0x0007, 
    0x00D8, 0x00DB, 0x00FB, 0x00E3, 0x00FA, 0x00E9, 0x00CF, 0x00C5, 
    0x0000, 0x0010, 0x0030, 0x002A, 0x0030, 0x0022, 0x0026, 0x002E, 
    0x0090, 0x008A, 0x00AA, 0x00AB, 0x00A2, 0x00A8, 0x00AE, 0x00AD, 
    0x01B0, 0x01BA, 0x01BE, 0x01A7, 0x01B6, 0x01AC, 0x018E, 0x0185, 
    0x01FA, 0x01E9, 0x01EF, 0x01E5, 0x01EC, 0x01ED, 0x01CD, 0x01C5, 
    0x0000, 0x0018, 0x0018, 0x0000, 0x0010, 0x0008, 0x0028, 0x0022, 
    0x00D8, 0x00C8, 0x00C8, 0x00C3, 0x00C8, 0x00CA, 0x00EA, 0x00E3, 
    0x00D8, 0x00D0, 0x00F0, 0x00E3, 0x00F8, 0x00E2, 0x00C6, 0x00C7, 
    0x00D8, 0x00C3, 0x00E3, 0x00E3, 0x00EA, 0x00E1, 0x00C7, 0x00C5, 
    0x0090, 0x0088, 0x00A8, 0x00A2, 0x00A0, 0x00AA, 0x00AE, 0x00A6, 
    0x00C8, 0x00CA, 0x00EA, 0x00E3, 0x00EA, 0x00E8, 0x00EE, 0x00E5, 
    0x01F8, 0x01E2, 0x01E6, 0x01E7, 0x01EE, 0x01E4, 0x01C6, 0x01C5, 
    0x016A, 0x0161, 0x0167, 0x0165, 0x016C, 0x0165, 0x0145, 0x0145, 
    0x0000, 0x0000, 0x0000, 0x0010, 0x0000, 0x0010, 0x0030, 0x0022, 
    0x0000, 0x0010, 0x0010, 0x000B, 0x0010, 0x0002, 0x0022, 0x002B, 
    0x0000, 0x0010, 0x0030, 0x003B, 0x0030, 0x0032, 0x0016, 0x000F, 
    0x0090, 0x009B, 0x00BB, 0x00AB, 0x00B2, 0x00A9, 0x008F, 0x008D, 
    0x0000, 0x0010, 0x0030, 0x0022, 0x0030, 0x0022, 0x0026, 0x0026, 
    0x0090, 0x0082, 0x00A2, 0x00AB, 0x00A2, 0x00A0, 0x00A6, 0x00AD, 
    0x01B0, 0x01B2, 0x01B6, 0x01AF, 0x01B6, 0x01A4, 0x0186, 0x018D, 
    0x0132, 0x0129, 0x012F, 0x012D, 0x0124, 0x012D, 0x010D, 0x010D, 
    0x0000, 0x0010, 0x0010, 0x0008, 0x0010, 0x0000, 0x0020, 0x002A, 
    0x0090, 0x0088, 0x0088, 0x008B, 0x0080, 0x008A, 0x00AA, 0x00AB, 
    0x0090, 0x0098, 0x00B8, 0x00A3, 0x00B0, 0x00AA, 0x008E, 0x0087, 
    0x0058, 0x004B, 0x006B, 0x0063, 0x006A, 0x0069, 0x004F, 0x0045, 
    0x0090, 0x0080, 0x00A0, 0x00AA, 0x00A0, 0x00A2, 0x00A6, 0x00AE, 
    0x0000, 0x000A, 0x002A, 0x002B, 0x0022, 0x0028, 0x002E, 0x002D, 
    0x0130, 0x012A, 0x012E, 0x0127, 0x0126, 0x012C, 0x010E, 0x0105, 
    0x016A, 0x0169, 0x016F, 0x0165, 0x016C, 0x016D, 0x014D, 0x0145, 
    0x0000, 0x0010, 0x0030, 0x0028, 0x0030, 0x0020, 0x0000, 0x000A, 
    0x0090, 0x0088, 0x00A8, 0x00AB, 0x00A0, 0x00AA, 0x008A, 0x008B, 
    0x01B0, 0x01B8, 0x0198, 0x0183, 0x0190, 0x018A, 0x018E, 0x0187, 
    0x01F8, 0x01EB, 0x01CB, 0x01C3, 0x01CA, 0x01C9, 0x01CF, 0x01C5, 
    0x01B0, 0x01A0, 0x01A0, 0x01AA, 0x01A0, 0x01A2, 0x0186, 0x018E, 
    0x01A0, 0x01AA, 0x01AA, 0x01AB, 0x01A2, 0x01A8, 0x018E, 0x018D, 
    0x01B0, 0x01AA, 0x018E, 0x0187, 0x0186, 0x018C, 0x018E, 0x0185, 
    0x016A, 0x0169, 0x014F, 0x0145, 0x014C, 0x014D, 0x014D, 0x0145, 
    0x0090, 0x0088, 0x00A8, 0x00A0, 0x00A0, 0x00A8, 0x0088, 0x0082, 
    0x00C8, 0x00C8, 0x00E8, 0x00E3, 0x00E8, 0x00EA, 0x00CA, 0x00C3, 
    0x01F8, 0x01E0, 0x01C0, 0x01C3, 0x01C8, 0x01C2, 0x01C6, 0x01C7, 
    0x0168, 0x0163, 0x0143, 0x0143, 0x014A, 0x0141, 0x0147, 0x0145, 
    0x01A0, 0x01A8, 0x01A8, 0x01A2, 0x01A0, 0x01AA, 0x018E, 0x0186, 
    0x0168, 0x016A, 0x016A, 0x0163, 0x016A, 0x0168, 0x014E, 0x0145, 
    0x0168, 0x0162, 0x0146, 0x0147, 0x014E, 0x0144, 0x0146, 0x0145, 
    0x016A, 0x0161, 0x0147, 0x0145, 0x014C, 0x0145, 0x0145, 0x0145
};



/* FUNCTION DEFINITIONS */


/**
 * Measures the time taken for a function to complete
 * @param fn the function to test
 * @return How many seconds have elapsed.
 */
double measure_time( TestFunc fn) {
    using namespace std::chrono;

    auto t1 = std::chrono::steady_clock::now();
    (*fn)();
    auto t2 = std::chrono::steady_clock::now();
    duration<double> time_span = duration_cast<duration<double>>(t2 - t1);
    return time_span.count();
}


/**
 * Print an error and exit
 * @param msg message to print before exiting
 */
void err(const char *msg) {
    perror(msg);
    exit(1);
}


/**
 * Print an error message, display usage and terminate
 * @param msg message to print
 */
void cmd_err(const char *msg) {
    printSignon();
    fprintf(stderr,"%s\n",msg);
    usage();
}


/**
 * Print program sign-on if it hasn't already been printed.
 */
void printSignon(void) {
    int     tempErrno;

    if (!signonPrinted) {
        tempErrno = errno;
        printf("Life (" VER_STRING ", " USE_BITMAP_STRING ", " \
            USE_FLOATS_STRING " build)\n");
        signonPrinted = 1;
        errno = tempErrno;
    }
}


/**
 * Print program usage then terminate.
 */
void usage(void) {
    extUsage(0);
}


/**
 * Print program usage then terminate.
 * @param extended if TRUE prints extended usage info
 */
void extUsage(int extended) {
    FILE *outfile = stdout;

    printSignon();
    fprintf(stdout, 
        "usage1: life [-v] [-s] [-w] [-l|-m] [d delay] [-g giffile] ticks " \
            "infile\n");
    if (extended) {
        fprintf(outfile, "\n" \
            "-d delay     Delay between gif frames, measured in MS\n" \
            "               (default: %" PRIu64 ")\n" \
            "-g giffile   Name of gif file to produce\n" \
            "-l           Force lex parsing\n" \
            "-m           Force map parsing\n" \
            "-s           Silent operation.  No output unless error occurs\n" \
            "-v           Verbose output\n" \
            "-w           Display warnings\n" \
            "ticks        How many ticks to run the simulation for\n" \
            "infile       Name of the input file to act as the seed\n" \
            "\n",
            gifFrameDelay
        );
    }
    fprintf(outfile,
        "usage2: life -r [-a seed] [-i init%%] [-l life%%] " \
            "[-t death%%]\n" \
            "             [-g giffile] [-d delay] [-v] [-s] ticks width " \
            "height\n");
    if (extended) {
        fprintf(outfile, "\n" \
            "-r           Generates a random map\n" \
            "-a seed      Seed for random number generator (default: %" \
                PRIu64 ")\n" \
            "-i init%%     Initial chance for a cell to be alive (default: "
                "%6.3f)\n" \
            "-l life%%     Chance per tick for each dead cell to come alive " \
                "(default: %6.3f)\n" \
            "-t death%%    Chance per tick for each live cell to die (default:"\
                " %6.3f)\n" \
            "-d delay     Delay between gif frames, measured in MS\n" \
            "               (default: %" PRIu64 ")\n" \
            "-g giffile   Name of gif file to produce\n" \
            "-s           Silent operation.  No output unless error occurs\n" \
            "-v           Verbose output\n" \
            "-w           Display warnings\n" \
            "width        width of the grid\n" \
            "height       height of the grid\n" \
            "ticks        number of ticks to run the simulation for\n" \
            "\n",
            randSeed,initLiveChance,randLifeChance,randDeathChance,gifFrameDelay
        );
    }
    if (extended) {
        fprintf(outfile,
            "-- Verification options --\n" \
            "life -t                      Test generator code for " \
                "verification\n" \
            "life -c                      Generate 'correct' values for "\
                "testing\n"
            "life -p [-l|-m] input_file   Parses the input file, prints it out"\
                " and quits\n" \
            "life -r -p [-a] [-i init%%] width height\n              " \
            "               Generates a random grid, prints it out and quits\n"
             );
    }
    exit(1);
}


/**
 * Reads a single character from the file buffer and updates the fileLineNo and
 * fileLinePos.
 * @return the character read
 * @note Any char less than CHAR_SPACE, other than CHAR_EOL, is returned as 0
 * @note Any read past the end of buffer will return (char)0
 */
unsigned char readFileChar(ParseState *ps) {
    unsigned char ch;
    if (ps->fileBufPos < fileBufSize) {
        ch = fileBuf[ps->fileBufPos++];
        if (ch == CHAR_EOL) {
            ps->fileLinePos = 1;
            ps->fileLineNo++;
        } else if (ch < CHAR_SPACE) {
            ch = (char)0;
        } else {
            ps->fileLinePos++;
        }
    } else {
        ch = (char)0;
    }
    return ch;
}


/**
 * Gets a character from the file buffer at the specified offset.  Does not
 * change the fileLineNo or fileLinePos.
 * @note Any char less than CHAR_SPACE is returned as 0
 * @note Any read past the end of buffer will return (char)0
 */
unsigned char getFileChar(ParseState *ps, uint64_t offset) {
    unsigned char ch;

    if (offset >= ps->fileBufSize) {
        ch = (char)0;
    } else {
        ch = fileBuf[offset];
        if (ch < CHAR_SPACE) {
            ch = (char)0;
        }
    }
    return ch;
}


/**
 * Reads the entire input file to 'fileBuf' and sets its size to 'fileBufSize'.
 * @param fn the name of the file to read
 */
void readInputFile(char *fn) {
    FILE *f;
    long fileSize;
    
    if ((f=fopen(fn,"rb")) == NULL) {
        err("Couldn't open input file");
    }
    
    if ((fseek(f,0,SEEK_END) != 0) ||
        ((fileSize = ftell(f)) == -1) ||
        (fseek(f,0,SEEK_SET) != 0)) {
        err("Problem determining input file size");
    }

    if (fileSize==0) {
        err("Input file is empty");
    }
    
    if (fileSize > MAX_INPUT_FILE_SIZE) {
        err("Input file is too large");
    }

    if ((fileBuf=(char*)malloc((size_t)fileSize)) == NULL) {
        err("Not enough memory");
    }
    fileBufSize = (unsigned long)fileSize;

    if (fread(fileBuf,1,(size_t)fileSize,f) != (size_t)fileSize) {
        err("Problem reading from input file");
    }
    fclose(f);

}


#if (ALLOW_GIF != 0)
/**
 * Writes a single frame describing the current grid to the gif file
 * @param g the gif writer object
 * @param pframe the frame to draw the grid to
 */
void writeGifFrame(GifWriter *g, uint8_t *pframe) {
    uint32_t    *pOut, x, y;

    pOut = (uint32_t*) &pframe[0];
    for(y=1; y <= gridHeight; y++) {
        for(x=1; x <= gridWidth; x++) {
            *pOut = (getCell(x,y)?CELL_COL_ALIVE:CELL_COL_DEAD);
            pOut++;
        }
    }   

    GifWriteFrame(g, pframe, gridWidth, gridHeight, gifFrameDelay);
}
#endif


/**
 * Allocates and initialises a grid buffer to the size indicated by (gridHeight 
 * x gridWidth).  It also points curGrid and nextGrid at grid[0].
 */
void initGrids(void) {
    int i;

    assert(gridHeight > 0);
    assert(gridWidth > 0);

#if (USE_BITMAP) 
    // how many GRID_UNITS to represent one row of cells?
    unitsPerRow = ((gridWidth+2) / 16) + ((gridWidth+2)%16?1:0);
    // multiply by number of rows +2
    // (the +1 is so the addressing in runLife_bm doesn't run over the
    //    end of the buffer)
    gridBufSize = unitsPerRow * (gridHeight + 2)+1;
    // convert gridBufSize to bytes
    gridBufSize *= sizeof(GRID_UNITS);
#else
    gridBufSize = (gridHeight + 2) * (gridWidth + 2);
#endif

    if (outOpts.verbose) {
        printf(
            "Allocating %u grids\n" \
            "    Dimensions: %" PRIu64 " x %" PRIu64 "\n"
            "    Bufer size: %" PRIu64 " bytes per grid\n",
            NUM_GRIDS, gridWidth, gridHeight, gridBufSize);
    }
    for(i=0; i<NUM_GRIDS; i++) {
        if (NULL == (grids[i] = (GRID_UNITS*)malloc(gridBufSize))) {
            err("Couldn't allocate grid");
        }
        memset(grids[i], 0, gridBufSize);
    }
    curGrid = grids[0];
    nextGrid = grids[0];
}


/**
 * Set the status (alive or dead) of a single cell in the grid
 * @param x the x-coord of the cell
 * @param y the y-coord of the cell
 * @param val the new status of the cell.  Non-zero for alive, 0 for dead
 */
void setCell_ch(uint64_t x, uint64_t y, int val) {
#ifdef DEBUG
    if ((x==0) || (x>gridWidth)) {
        fprintf(stderr,"setCell: x out of bounds (Accessing %" PRIu64 \
            ":%" PRIu64 ")",x,y);
        exit(1);
    }
    if ((y==0) || (y>gridHeight)) {
        fprintf(stderr,"getCell: y out of bounds (Accessing %" PRIu64 \
            ":%" PRIu64 ")",x,y);
        exit(1);
    }
#endif
    nextGrid[y*(gridWidth+2)+x] = (val>0);
}


/**
 * Get the status (alive or dead) of a single cell in the grid
 * @param x the x-coord of the cell
 * @param y the y-coord of the cell
 * @return 1 if the cell is alive, 0 if it is dead
 */
unsigned char getCell_ch(uint64_t x, uint64_t y) {
#ifdef DEBUG
   if(x>gridWidth+1) {
        fprintf(stderr,"getCell: x out of bounds (Accessing %" PRIu64 \
            ":%" PRIu64 ")",x,y);
        exit(1);
    }
    if (y>gridHeight+1) {
        fprintf(stderr,"getCell: y out of bounds (Accessing %" PRIu64 \
            ":%" PRIu64 ")",x,y);
        exit(1);
    }
#endif
    return curGrid[y*(gridWidth+2)+x];
} 


/**
 * Get the status (alive or dead) of a single cell in the grid
 * @param g the grid to get the cell from
 * @param x the x-coord of the cell
 * @param y the y-coord of the cell
 * @return 1 if the cell is alive, 0 if it is dead
 */
unsigned char getCellFromGrid_ch(int g, uint64_t x, uint64_t y) {
#ifdef DEBUG
   if(x>gridWidth+1) {
        fprintf(stderr,"getCell: x out of bounds (Accessing %" PRIu64 \
            ":%" PRIu64 ")",x,y);
        exit(1);
    }
    if (y>gridHeight+1) {
        fprintf(stderr,"getCell: y out of bounds (Accessing %" PRIu64 \
            ":%" PRIu64 ")",x,y);
        exit(1);
    }
#endif
    return grids[g][y*(gridWidth+2)+x];
} 


/**
 * This function does the steps for generating the successive generations and
 * (optionally) creates a gif file.
 */
void runLife_ch(void) {
    uint64_t    tick, x, y;
    int         numLive, numLiveRight, nextGridNum = 1, old1, old2, newState;
#if (USE_FLOATS != 0)
    double      randChance;
#else
    uint64_t    randLifePerMil = randLifeChance*1000000,
                randDeathPerMil = randDeathChance*1000000,
                randMil;
#endif

    curGrid = grids[0];
    nextGrid = grids[1];
#if (ALLOW_GIF != 0)
    GifWriter   g;
    uint8_t     *frame = NULL;

    if (gifFn != NULL ) {
        if ((frame = (unsigned char*)malloc(gridHeight * gridWidth * 4)) 
                == NULL) {
            err("Memory allocation failed\n");
        }
        memset(frame, 0, gridHeight * gridWidth *4);
        GifBegin(&g, gifFn, gridWidth, gridHeight, gifFrameDelay);
        writeGifFrame(&g, frame);
    }
#endif
    if (outOpts.verbose) {
        printf("Ticks: %" PRIu64 "\n", numTicks);
        if (randGrid) {
            printf("Initial live: %f\nLive chance: %f\nDeath chance: %f\n",
                initLiveChance, randLifeChance, randDeathChance);
        }
    }
    for (tick=0; tick < numTicks; tick++) {
        for(y=1; y <= gridHeight; y++) {
            // prepare the old1, old2 'pipeline'
            old1 = getCell(1,y-1)+getCell(1,y)+getCell(1,y+1);
            old2 = 0;
            for(x=1; x <= gridWidth; x++) {
                // How many right-side neighbours are alive?
                numLiveRight = getCell(x+1,y-1) + getCell(x+1,y) 
                             + getCell(x+1,y+1);
                // Add the right-side neighbours to the top and bottom
                //   neighbours and the left side which we 'remember' from
                //   not the previous cell but the one before that.
                numLive = old2 + getCell(x,y-1) + getCell(x,y+1) + numLiveRight;
                old2 = old1;
                old1 = numLiveRight;
                // Yes, this 'if' statement could be done in one line but it
                //   would be a bugger to read and I suspect the compiler will
                //   produce the same code with this easier-to-read version. 
                if (getCell(x,y) > 0) {
                    // cell is alive
                    newState = (((numLive == 2) || (numLive == 3)) ? 1 : 0);
                } else {
                    // cell is dead
                    newState = ((numLive == 3) ? 1 : 0);
                }

                if (randGrid) {
#if (USE_FLOATS != 0)
                    // give me a number 0 <= n < 1
                    randChance = ((double)(rand()%100000))/100000;
                    if (newState) {
                        if (randChance < randDeathChance) {
                            newState = 0;
                        }
                    } else {
                        if (randChance < randLifeChance) {
                            newState = 1;
                        }
                    }

#else
                    randMil = (uint64_t)(rand()%1000000);
                    if (newState) {
                        if (randMil < randDeathPerMil) {
                            newState = 0;
                        }
                    } else {
                        if (randMil < randLifePerMil) {
                        }
                    }
#endif
                }
                setCell(x,y,newState);
            }
        }
#if (ALLOW_GIF != 0)
        if (gifFn != NULL) {
            writeGifFrame(&g, frame);
        }
#endif
        // curGrid becomes the next grid and nextGrid gets pointed to the 'grid'
        //   after the new 'curGrid'
        curGrid = grids[nextGridNum];
        nextGridNum++;
        if (nextGridNum >= NUM_GRIDS) {
            nextGridNum = 0;
        }
        nextGrid = grids[nextGridNum];
    }
#if (ALLOW_GIF != 0)
    if (gifFn != NULL) {
        GifEnd(&g);
        free(frame);
    }
#endif
}


/**
 * This function does the steps for generating the successive generations and
 * (optionally) creates a gif file.
 * WARNING: This function is not actually used.
 */
void runLife_ch_useoffsets(void) {
    uint64_t    tick, x, y;
    int         numLive, numLiveRight, nextGridNum = 1, old1, old2;
#if (USE_FLOATS != 0)
    double      randChance;
#else
    uint64_t    randLifePerMil = randLifeChance*1000000,
                randDeathPerMil = randDeathChance*1000000,
                randMil, 
                ofsUL,ofsU,ofsUR,ofsL,ofsR,ofsDL,ofsD,ofsDR,cellPos;
    unsigned char newState;
#endif

    curGrid = grids[0];
    nextGrid = grids[1];
#if (ALLOW_GIF != 0)
    GifWriter   g;
    uint8_t     *frame = NULL;

    if (gifFn != NULL ) {
        if ((frame = (unsigned char*)malloc(gridHeight * gridWidth * 4)) 
                == NULL) {
            err("Memory allocation failed\n");
        }
        memset(frame, 0, gridHeight * gridWidth *4);
        GifBegin(&g, gifFn, gridWidth, gridHeight, gifFrameDelay);
        writeGifFrame(&g, frame);
    }
#endif
    if (outOpts.verbose) {
        printf("Ticks: %" PRIu64 "\n", numTicks);
        if (randGrid) {
            printf("Initial live: %f\nLive chance: %f\nDeath chance: %f\n",
                initLiveChance, randLifeChance, randDeathChance);
        }
    }
    ofsUL = ((uint64_t)0)-(gridWidth+2)-1;
    ofsU = ofsUL + 1;
    ofsUR = ofsU + 1;
    ofsL = ((uint64_t)0) - 1;
    ofsR = ofsL + 2;
    ofsDL = ((uint64_t)0)+(gridWidth+2)-1;
    ofsD = ofsDL + 1;
    ofsDR = ofsD + 1;

    for (tick=0; tick < numTicks; tick++) {
        cellPos = (gridWidth+2)+1;
        for(y=1; y <= gridHeight; y++) {
            // prepare the old1, old2 'pipeline'
            old1 = curGrid[cellPos+ofsU] 
                 + curGrid[cellPos] 
                 + curGrid[cellPos+ofsD];
            old2 = 0;
            for(x=1; x <= gridWidth; x++) {
                // How many right-side neighbours are alive?
                numLiveRight = curGrid[cellPos+ofsUR] 
                             + curGrid[cellPos+ofsR] 
                             + curGrid[cellPos+ofsDR];
                // Add the right-side neighbours to the top and bottom
                //   neighbours and the left side which we 'remember' from
                //   not the previous cell but the one before that.
                numLive = old2 + curGrid[cellPos+ofsU] + curGrid[cellPos+ofsD] 
                        + numLiveRight;
                old2 = old1;
                old1 = numLiveRight;
                // Yes, this 'if' statement could be done in one line but it
                //   would be a bugger to read and I suspect the compiler will
                //   produce the same code with this easier-to-read version. 
                if (getCell(x,y) > 0) {
                    // cell is alive
                    newState = (((numLive == 2) || (numLive == 3)) ? 1 : 0);
                } else {
                    // cell is dead
                    newState = ((numLive == 3) ? 1 : 0);
                }

                if (randGrid) {
#if (USE_FLOATS != 0)
                    // give me a number 0 <= n < 1
                    randChance = ((double)(rand()%100000))/100000;
                    if (newState) {
                        if (randChance < randDeathChance) {
                            newState = 0;
                        }
                    } else {
                        if (randChance < randLifeChance) {
                            newState = 1;
                        }
                    }
#else
                    randMil = (uint64_t)(rand()%1000000);
                    if (newState) {
                        if (randMil < randDeathPerMil) {
                            newState = 0;
                        }
                    } else {
                        if (randMil < randLifePerMil) {
                        }
                    }
#endif
                }
                nextGrid[cellPos] = newState;
                cellPos++; // next cell
            } // end of for(x loop
            cellPos += 2; // skip the padding cells around the edge of the grid
        } // end of for(y  loop
#if (ALLOW_GIF != 0)
        if (gifFn != NULL) {
            writeGifFrame(&g, frame);
        }
#endif
        // curGrid becomes the next grid and nextGrid gets pointed to the 'grid'
        //   after the new 'curGrid'
        curGrid = grids[nextGridNum];
        nextGridNum++;
        if (nextGridNum >= NUM_GRIDS) {
            nextGridNum = 0;
        }
        nextGrid = grids[nextGridNum];
    }
#if (ALLOW_GIF != 0)
    if (gifFn != NULL) {
        GifEnd(&g);
        free(frame);
    }
#endif
}


/**
 * Set the status (alive or dead) of a single cell in the grid
 * @param x the x-coord of the cell
 * @param y the y-coord of the cell
 * @param val the new status of the cell.  Non-zero for alive, 0 for dead
 */
void setCell_bm(uint64_t x, uint64_t y, int val) {
#ifdef DEBUG
    if ((x==0) || (x>gridWidth)) {
        fprintf(stderr,"setCell: x out of bounds (Accessing %" PRIu64 \
            ":%" PRIu64 ")",x,y);
        exit(1);
    }
    if ((y==0) || (y>gridHeight)) {
        fprintf(stderr,"getCell: y out of bounds (Accessing %" PRIu64 \
            ":%" PRIu64 ")",x,y);
        exit(1);
    }
#endif
    GRID_UNITS *p;
    p = &nextGrid[y*unitsPerRow+(x/16)];
    if (p < nextGrid) {
        printf("p too low\n");
    }
    if (p > &nextGrid[gridBufSize/sizeof(GRID_UNITS)]) {
        printf("p too high\n");
    }

    *p = (*p & (0xFFFF ^ (1 << (x%16)))) + (val?1<<(x%16):0);
}


/**
 * Get the status (alive or dead) of a single cell in the grid
 * @param x the x-coord of the cell
 * @param y the y-coord of the cell
 * @return 1 if the cell is alive, 0 if it is dead
 */
unsigned char getCell_bm(uint64_t x, uint64_t y) {
#ifdef DEBUG
   if(x>gridWidth+1) {
        fprintf(stderr,"getCell: x out of bounds (Accessing %" PRIu64 \
            ":%" PRIu64 ")",x,y);
        exit(1);
    }
    if (y>gridHeight+1) {
        fprintf(stderr,"getCell: y out of bounds (Accessing %" PRIu64 \
            ":%" PRIu64 ")",x,y);
        exit(1);
    }
#endif
    return ((curGrid[y*unitsPerRow+(x/16)] & (1 << (x%16))) > 0 ? 1 : 0);
} 


/**
 * Get the status (alive or dead) of a single cell in the grid
 * @param g the grid to get the cell from
 * @param x the x-coord of the cell
 * @param y the y-coord of the cell
 * @return 1 if the cell is alive, 0 if it is dead
 */
unsigned char getCellFromGrid_bm(int g, uint64_t x, uint64_t y) {
#ifdef DEBUG
   if(x>gridWidth+1) {
        fprintf(stderr,"getCell: x out of bounds (Accessing %" PRIu64 \
            ":%" PRIu64 ")",x,y);
        exit(1);
    }
    if (y>gridHeight+1) {
        fprintf(stderr,"getCell: y out of bounds (Accessing %" PRIu64 \
            ":%" PRIu64 ")",x,y);
        exit(1);
    }
#endif
    return ((grids[g][y*unitsPerRow+(x/16)] & (1 << (x%16))) > 0 ? 1 : 0);
} 


/**
 * This function does the steps for generating the successive generations and
 * (optionally) creates a gif file.
 */
void runLife_bm(void) {
    uint64_t    tick, x, y;
    GRID_UNITS  mask, *p;
    int         numLive, nextGridNum = 1, newState, shift, fix;
#if (USE_FLOATS != 0)
    double      randChance;
#else
    uint64_t    randLifePerMil = randLifeChance*1000000,
                randDeathPerMil = randDeathChance*1000000,
                randMil;
#endif
     // each offset in 'lut' is how many bits in that offset are 1's.
     //   eg. 0 has 0 bits set, the number 1 (0001b), 2 (0010b) has 1 bit set, 
     //   3 (0011b) has 2 bits set, etc
    int         lut[] = {0,1,1,2,1,2,2,3};

    curGrid = grids[0];
    nextGrid = grids[1];
#if (ALLOW_GIF != 0)
    GifWriter   g;
    uint8_t     *frame = NULL;

    if (gifFn != NULL ) {
        if ((frame = (unsigned char*)malloc(gridHeight * gridWidth * 4)) 
                == NULL) {
            err("Memory allocation failed\n");
        }
        memset(frame, 0, gridHeight * gridWidth *4);
        GifBegin(&g, gifFn, gridWidth, gridHeight, gifFrameDelay);
        writeGifFrame(&g, frame);
    }
#endif
    if (outOpts.verbose) {
        printf("Ticks: %" PRIu64 "\n", numTicks);
        if (randGrid) {
            printf("Initial live: %f\nLive chance: %f\nDeath chance: %f\n",
                initLiveChance, randLifeChance, randDeathChance);
        }
    }
    for (tick=0; tick < numTicks; tick++) {
        for(y=1; y <= gridHeight; y++) {
            for(x=1; x <= gridWidth; x++) {
                // We're interested in 3 consective bits in each row.  If those
                //   bits would span two int16_t's, then we'd have to do two
                //   lots of masking and shifting.  That would be no fun!
                //   Instead we'll point at the int16_t one byte after the one
                //   we 'should' be looking at, and then 'fix' the 'mask' and
                //   'shift' variables accordingly.
                fix = ((x-1)%16>=8);
                shift = ((x-1)%16) - (fix?8:0);
                mask = 0x7 << shift;
                p = (GRID_UNITS*) 
                    ((char*)(&curGrid[((y-1)*unitsPerRow)+((x-1)/16)])
                        +(fix?1:0));
                numLive = 
                       lut[(*p&mask)>>shift]
                       // 0x5 is to mask out the cell we're currently looking at
                       // We only want to count that cell's neighbours
                     + lut[(((p[unitsPerRow]&mask))>>shift) & 0x5] 
                     + lut[(p[2*unitsPerRow]&mask)>>shift];

                // Yes, this 'if' statement could be done in one line but it
                //   would be a bugger to read and I suspect the compiler will
                //   produce the same code with this easier-to-read version. 
                if (((((p[unitsPerRow]&mask))>>shift) & 0x2) > 0) {
                    // cell is alive
                    newState = (((numLive == 2) || (numLive == 3)) ? 1 : 0);
                } else {
                    // cell is dead
                    newState = ((numLive == 3) ? 1 : 0);
                }

                if (randGrid) {
#if (USE_FLOATS != 0)
                    // give me a number 0 <= n < 1
                    randChance = ((double)(rand()%100000))/100000;
                    if (newState) {
                        if (randChance < randDeathChance) {
                            newState = 0;
                        }
                    } else {
                        if (randChance < randLifeChance) {
                            newState = 1;
                        }
                    }

#else
                    randMil = (uint64_t)(rand()%1000000);
                    if (newState) {
                        if (randMil < randDeathPerMil) {
                            newState = 0;
                        }
                    } else {
                        if (randMil < randLifePerMil) {
                        }
                    }
#endif
                }
                setCell(x,y,newState);
            }
        }
#if (ALLOW_GIF != 0)
        if (gifFn != NULL) {
            writeGifFrame(&g, frame);
        }
#endif
        // curGrid becomes the next grid and nextGrid gets pointed to the 'grid'
        //   after the new 'curGrid'
        curGrid = grids[nextGridNum];
        nextGridNum++;
        if (nextGridNum >= NUM_GRIDS) {
            nextGridNum = 0;
        }
        nextGrid = grids[nextGridNum];
    }
#if (ALLOW_GIF != 0)
    if (gifFn != NULL) {
        GifEnd(&g);
        free(frame);
    }
#endif
}


/**
 * Tests the 'runLife' function with every 3x3 grid to verify that it produces
 * the correct result.  The output is compared against the 'correct' array.
 * See the comments accompanying the declaration of the 'correct' array above
 * for more details on the 'correct' values.
 */
void runTest(void) {
    int i, x, y, test;
    char on='O',off='.';
    uint16_t result;

    gridHeight=3;
    gridWidth=3;

    initGrids();
    
    for(test=0; test<512; test++) {
//    for(test=7;test<8;test++) {
        i = test;
        nextGrid = grids[0];
        curGrid = nextGrid;

        for(y=1; y<=3; y++) {
            for(x=1; x<=3; x++) {
                setCell(x,y,(i&1));
                i >>= 1;
            }
        }

        numTicks = 1;
        runLife();

        printf("test %id  (input: %04Xh  expect: %04Xh)\n",
            test, test, correct[test]);

        for(y=1; y<=3; y++) {
            printf("%c%c%c   %c%c%c\n",
                (getCellFromGrid(0,1,y)?on:off),
                (getCellFromGrid(0,2,y)?on:off),
                (getCellFromGrid(0,3,y)?on:off),
                (getCellFromGrid(1,1,y)?on:off),
                (getCellFromGrid(1,2,y)?on:off),
                (getCellFromGrid(1,3,y)?on:off)
                );
        }
        printf("\n");

        result = 0;
        for(y=1; y<=3;y++) {
            for(x=1; x<=3; x++) {
                result = (result >> 1) + (getCellFromGrid(1,x,y)>0?0x100:0);
            }
        }
        if (result != correct[test]) {
            printf("Test %i failed!\n",test);
            printf("result:  %04Xh\n",result); 
            printf("correct: %04Xh\n",correct[test]);
            exit(1);
        }
    }
    printf("All tests passed.\n");
    exit(0);
}


/**
 * Generates output describing the correct output for every 3x3 grid for
 * redirecting to file which, in turn, should go back into this source.
 * See the comments accompanying the declaration of the 'correct' array above
 * for more details on the 'correct' values.
 */
void generateCorrect(void) {
    int i, x, y, test;
    uint16_t result;
 
    gridHeight=3;
    gridWidth=3;

    initGrids();
    
    printf("uint16_t results[] = {\n");
    for(test=0; test<512; test++) {

        i = test;
        for(y=1; y<=3; y++) {
            for(x=1; x<=3; x++) {
                setCell(x,y,(i&1));
                i >>= 1;
            }
        }

        numTicks = 1;
        runLife();

        result = 0;
        for(y=1; y<=3;y++) {
            for(x=1; x<=3; x++) {
                result = (result >> 1) + (getCellFromGrid(1,x,y)>0?0x100:0);
            }
        }
        printf("0x%04X",result);
        if (test <511) printf(", ");
        if (test % 8 == 7) printf("\n");
    }
    printf("};\n");
    exit(0);
}


/**
 * Parses a double (floating point) value from a string
 * @param str the string to convert
 * @param pval the buffer where the result should be stored
 * @return TRUE if the conversion was successful, FALSE otherwise
 */
bool parseDouble(const char *str, double *pval) {
    double d;
    char *p = (char*)str, *end;
    bool nonsp=false;

    if (str == NULL) {
        return false;
    }
    while (*p && (!nonsp)) {
        if (!isspace(*p)) {
            nonsp = true;
        }
    }
    if (nonsp == false) {
        return false;
    }
    errno = 0;
    d = strtod(str,&end);
    if (errno != 0) {
        return false;
    }
    if (*end != '\0') {
        return false;
    }
    *pval = d;
    return true;
}


/**
 * Parses a 64-bit unsigned int from a hex string
 * @param str the string to convert
 * @param pval the buffer where the result should be stored
 * @return TRUE if the conversion was successful, FALSE otherwise
 */
bool parseUHex64(const char *buf, uint64_t *pval) {
    char        *p = (char*)buf;
    uint64_t    tmp=0;

    if ((buf == NULL) || (strlen(buf) > 128)) {
        return false;
    }
    while (*p) {
        tmp <<= 4;
        if ((*p >= '0') && (*p <= '9')) {
            tmp += *p - '0';
            p++;
            continue;
        }
        if ((*p >= 'a') && (*p <= 'f')) {
            tmp += *p - 'a' + 10;
            p++;
            continue;
        }
        if ((*p >= 'A') && (*p <= 'F')) {
            tmp += *p - 'a' + 10;
            p++;
            continue;
        }
        return false;
    }
    *pval = tmp;
    return true;
}


/**
 * Parses a 64-bit unsigned integer from a string
 * @param str the string to convert
 * @param pval the buffer where the result should be stored
 * @return TRUE if the conversion was successful, FALSE otherwise
 */
bool parseUint64(const char *buf, uint64_t *pval) {
    uint64_t tmp;


    errno = 0;
    if (((tmp = strtoull(buf,NULL,10)) == 0) ||
        ((tmp == ULLONG_MAX) && (errno == ERANGE))) {
        return false;
    }
    *pval = tmp;
    return true;
}


/**
 * Parses the command line using getopt and sets the appropriate variables
 * @param argc number of command line arguments
 * @param argv an array of pointers to argv[0] through argv[argc-1]
 * @param pout the output options for this program
 */
void parseCmdLineDef(int argc, char**argv, OutputOptions *pout) {
    int opt;

    while ((opt = getopt(argc, argv, "pPsSlLmMvVwWd:D:g:G:")) != -1) {
        switch(opt) {
            case 'p':
            case 'P':
                pout->printGridOnly = true;
                break;

            case 's':
            case 'S':
                pout->silent = true;
                break;

            case 'l':
            case 'L':
                if (forceMapParse) {
                    cmd_err("Exclusive args specified '-l' and '-m'");
                }
                forceLexParse = true;
                break;

            case 'm':
            case 'M':
                if (forceLexParse) {
                    cmd_err("Exclusive args specified '-l' and '-m'");
                }
                forceMapParse = true;
                break;

            case 'v':
            case 'V':
                pout->verbose = true;
                break;

            case 'g':
            case 'G':
                gifFn = optarg;
                break;
            
            case 'w':
            case 'W':
                pout->showWarns = true;
                break;

            case 'd':
            case 'D':
                if ((parseUint64(optarg,&gifFrameDelay) == false) ||
                    (gifFrameDelay>=100000)) {
                    cmd_err("Invalid value for 'delay' argument " \
                        "(1-99999 allowed)");
                }
                break;

            default:
                usage();
            
        }
    }
    if (pout->printGridOnly) {
        optind--; // with "-P" there should not be a 'ticks' argument
    } else {
        if (optind + 1 > argc) {
            cmd_err("Missing 'ticks' argument");
        }
        if (parseUint64(argv[optind],&numTicks) == false) {
            cmd_err("Invalid value for 'ticks' argument.");            
        }
    }

    if (optind + 2 > argc) {
        cmd_err("Missing 'in_file' argument");
    }
    if (argc > optind + 2) {
        cmd_err("Extra argument(s) after 'in_file'");
    }
    inFn = argv[optind+1];
}


/**
 * Parses the command line for a random grid
 * @param argc number of command line arguments
 * @param argv an array of pointers to argv[0] through argv[argc-1]
 * @param pout the output options for this program
 */
void parseCmdLineRand(int argc, char**argv, OutputOptions *pout) {
    int opt;

    while ((opt = getopt(argc, argv, "pPsSwWvVrRa:A:i:I:l:L:d:D:t:T:g:G:")) 
            != -1) {
        switch(opt) {
            case 'p':
            case 'P':
                pout->printGridOnly = true;
                break;

            case 'w':
            case 'W':
                pout->showWarns = true;
                break;

            case 'v':
            case 'V':
                pout->verbose = true;
                break;

            case 's':
            case 'S':
                pout->silent = true;
                break;

            case 'r':
            case 'R':
                randGrid = true;
                break;

            case 'i':
            case 'I':
                if ( (parseDouble(optarg, &initLiveChance) == false) ||
                    (initLiveChance <= 0) || (initLiveChance >= 1)) {
                    cmd_err("Invalid 'init_chance' value (0 < n < 1)");
                }
                break;
            
            case 'a': 
            case 'A':
                if (parseUint64(optarg,&randSeed) == false) {
                    cmd_err("Invalid 'rand_seed'");
                }
                break;

            case 'g':
            case 'G':
                gifFn = optarg;
                break;
            
            case 't':
            case 'T':
                if ( (parseDouble(optarg, &randDeathChance)== false) ||
                    (randDeathChance < 0) || (randDeathChance >= 1)) {
                    cmd_err("Invalid 'death_chance' value (0 <= n < 1)");
                }
                break;

            case 'l':
            case 'L':
                if ( (parseDouble(optarg, &randLifeChance)==false) ||
                    (randLifeChance < 0) || (randLifeChance >= 1)) {
                    cmd_err("Invalid 'life_chance' value (0 <= n < 1)");
                }
                break;

            case 'd':
            case 'D':
                if ((parseUint64(optarg,&gifFrameDelay) == false) ||
                    (gifFrameDelay>=100000)) {
                    cmd_err("Invalid value for 'delay' argument " \
                        "(1-99999 allowed)");
                }
                break;

            default:
                fprintf(stderr,"Unknown arg: %s\n",argv[optind]);
                usage();
            
        }
    }
    if (pout->printGridOnly) {
        optind--; // with "-P" there should not be a 'ticks' argument
    } else {
        if (optind + 1 > argc) {
            cmd_err("Missing 'ticks' argument");
        }
        if (parseUint64(argv[optind], &numTicks) == false) {
            cmd_err("Invalid 'ticks' argument");
        }
    }
    if (optind + 2 > argc) {
        cmd_err("Missing 'width' argument");
    }
    if (parseUint64(argv[optind+1], &gridWidth) == false) {
        cmd_err("Invalid 'width' argument");
    }
    if (optind + 3 > argc) {
        cmd_err("Missing 'height' argument");
    }
    if (parseUint64(argv[optind+2], &gridHeight) == false) {
        cmd_err("Invalid 'height' argument");
    }
    if (argc > optind + 3) {
        cmd_err("Extra argument after 'height'");
    }
}


/**
 * Prints the grid to stdout
 */
void printGrid(void) {
    unsigned int x,y;
    const char *aliveChars = LEX_CHARS_ALIVE,
                *deadChars = LEX_CHARS_DEAD;

    for(y=1; y<=gridHeight; y++) {
        for(x=1; x<=gridWidth; x++) {
            printf("%c",getCell(x,y)?aliveChars[0]:deadChars[0]);
        }
        printf("\n");
    }
}


/**
 * Populates a random grid according to the initial life chance
 */
void populateRandGrid(void) {
    uint64_t x,y;
    float r;

    initGrids();
    nextGrid = grids[0];
    for(y=1; y<=gridHeight; y++) {
        for(x=1; x<=gridWidth; x++) {
            r = ((float)(rand()%100000)) / 100000;
            if (r <= initLiveChance) {
                setCell(x,y,1);
            }
        }
    }
}


/**
 * Parses the command line for a random grid
 * @param argc number of command line arguments
 * @param argv an array of pointers to argv[0] through argv[argc-1]
 * @param pout the output options for this program
 */
void parseCmdLine(int argc, char **argv, OutputOptions *pout) {
    if (argc==1) {
        extUsage(1);
    }

    if ((argc==2) && ((strcmp(argv[1],"-t")==0) || (strcmp(argv[1],"-T")==0))){
        runTest();
    }

    if ((argc==2) && ((strcmp(argv[1],"-c")==0) || (strcmp(argv[1],"-c")==0))){
        generateCorrect();
    }

    if ((strcmp(argv[1],"-r")==0) || (strcmp(argv[1],"-R")==0)) {
        randGrid = true;
        parseCmdLineRand(argc,argv,pout);
    } else {
        randGrid = false;
        parseCmdLineDef(argc,argv,pout);
    }

    if (pout->silent) {
        pout->verbose = false;
        pout->showWarns = false;
    }
}


/**
 * program entry point
 * @param argc number of command line arguments
 * @param argv an array of pointers to argv[0] through argv[argc-1]
 * @return exit code for the program
 */
int main(int argc, char **argv) {
    int             gridNo;
    double          elapsed;
    ParserInfo      pi;

    outOpts.silent = false;
    outOpts.printGridOnly = false;
    outOpts.verbose = false;
    outOpts.showWarns = false;
    parseCmdLine(argc, argv, &outOpts);

    if (!outOpts.silent) {
        printSignon();
        if (randGrid) {
            printf("Random grid\n");
        } else {
            printf("Input file: %s\n", inFn);
        }
    }
    if (outOpts.verbose) {
        if (gifFn != NULL) {
            printf("Writing gif\n    name: %s\n    frame delay: %" PRIu64 "\n",
                gifFn,gifFrameDelay);
        }
    }

    if (randGrid) {
        populateRandGrid();
    } else {
        // read the input file and parse it
        readInputFile(inFn);
        
        pi.doVerbose = outOpts.verbose;
        pi.showWarns = outOpts.showWarns;
        pi.fileBufSize = fileBufSize;
        pi.readCharFunc = readFileChar;
        pi.getCharFunc = getFileChar;
        pi.lexCharsAlive = LEX_CHARS_ALIVE;
        pi.lexCharsDead = LEX_CHARS_DEAD;
        pi.widthVal = &gridWidth;
        pi.heightVal = &gridHeight;
        pi.getCellFunc = getCell;
        pi.setCellFunc = setCell;
        pi.initGridFunc = initGrids;

        if (forceMapParse) {
            parseMap(&pi);
        }else if (forceLexParse) {
            parseLex(&pi);
        } else {
            parseByGuess(&pi);
        }
        free(fileBuf);
    }

    if (outOpts.printGridOnly) {
        printGrid();
        exit(0);
    }

    // outOpts.verbose=1;
    // randGrid = 1;
    // gridHeight = 200;
    // gridWidth = 200;
    // numTicks = 1;
    // initGrids();

    elapsed = measure_time(runLife);
    printf("Elapsed: %6.3f seconds\n",elapsed);

    for(gridNo=NUM_GRIDS-1; gridNo>=0; gridNo--) {
        if (grids[gridNo] != NULL) {
            free(grids[gridNo]);
        }
    }
}




