/*
 *  Written by: Brent Feulbach
 *  Date: 20-Aug-2020
 */

#ifndef INC_LIFE_H

/* INCLUDES */

/*
 * These macros allow one to printf fixed size values.  See:
 * https://en.cppreference.com/w/cpp/types/integer
 */ 
#define __STDC_LIMIT_MACROS
#define __STDC_CONSTANT_MACROS
#define __STDC_FORMAT_MACROS

#include <assert.h>
#include <chrono>
#include <ctype.h>
#include <errno.h>
#include <inttypes.h>
#include <limits.h>
#include <math.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>
#include <vector>

#include "gif.h"
#include "lifepars.h"


/* DEFINITIONS */

// There's two versions of this program's inner loop code; the 'bitmap' version
//   and the 'char'.
// The 'bitmap' version represents the cells with a bitmap and shifts and masks
//   bits to access the individual cells.
// The 'char' version represents each cell with one byte.
#define USE_BITMAP 0

// The random chance of a cell randomly coming alive or dying with the '-r'
//   command line option.  These chances are specified as floating point
//   numbers.  If use floats is 0 then those chances are converted to X in a
//   million and integer math is used to calculate whether these things happen
//   or not.
#define USE_FLOATS 0

#define ALLOW_GIF 1                     // allow gif to be written
                                        // grid at each frame
#define DEFAULT_GIF_DELAY 50            // delay between successive frames

/* For the pixel colours, the lowest byte is red (0 to 255).  The second lowest
 * byte is green (0-255) and third highest is blue (0-255).  The highest byte is
 * ignored.
 */
#define CELL_COL_ALIVE 0x00FFFFFFUL     // colour for a live cell
#define CELL_COL_DEAD 0UL               // colour for a dead cell

#define NUM_GRIDS 2                     // Number of grid buffers to use
#define MAX_INPUT_FILE_SIZE 2000000000  // 2 gigabyte

#define LEX_CHARS_ALIVE "Oo"            // these chars are live cells in a lex
#define LEX_CHARS_DEAD "._-"            // these chars are dead cells in a lex



/* TYPEDEFS */

// An arbirary function which will be called from 'measure_time'.
typedef void (*TestFunc)();



/* STRUCTS */

typedef struct tagOutputOptions {
    bool        verbose;
    bool        silent;
    bool        showWarns;
    bool        printGridOnly;
} OutputOptions;



/* MACROS */

// Use the correct set of functions for either bitmap or char representation.
#if (USE_BITMAP!=0)
    #define runLife runLife_bm
    #define getCell getCell_bm
    #define getCellFromGrid getCellFromGrid_bm
    #define setCell setCell_bm
    #define GRID_UNITS uint16_t
#else
    #define runLife runLife_ch
    #define getCell getCell_ch
    #define getCellFromGrid getCellFromGrid_ch
    #define setCell setCell_ch
    #define GRID_UNITS unsigned char
#endif


/* FUNCTION PROTOTYPES */

#ifdef __cplusplus
extern "C" {
#endif

void err(const char*);
void extUsage(int);
void generateCorrect(void);
unsigned char getCell_bm(uint64_t, uint64_t);
unsigned char getCell_ch(uint64_t, uint64_t);
unsigned char getCellFromGrid_bm(int, uint64_t, uint64_t);
unsigned char getCellFromGrid_ch(int, uint64_t, uint64_t);
unsigned char getFileChar(uint64_t);
void initGrids(void);
double measure_time(TestFunc);
void populateRandGrid(void);
void printGrid(void);
void printSignon(void);
unsigned char readFileChar(void);
void readInputFile(const char *);
void runLife_bm(void);
void runLife_ch(void);
void runTest(void);
void setCell_bm(uint64_t, uint64_t, int);
void setCell_ch(uint64_t, uint64_t, int);
void usage();
#if (ALLOW_GIF != 0)
    void writeGifFrame(GifWriter *, uint8_t*);
#endif

#ifdef __cplusplus
}
#endif



#define INC_LIFE_H
#endif