/*
 *  Written by: Brent Feulbach
 *  Date: 20-Aug-2020
 */

#include "lifepars.h"

/* FUNCTIONS */

/*
 * The parseErrXXX function below are all involved with printing error messages
 * for the parsing code.
 * 
 */ 



/**
 * Returns a string corresponding to a give 'expect' value.
 * @param expect the value to get a string for
 * @return a string corresponding to the param
 */
const char *expectString(int expect) {
    const char *s;

    switch(expect) {
        case EXPECT_NONE: s = "none"; break;
        case EXPECT_CLOSEBR: s = "]"; break;
        case EXPECT_COMMA: s = "comma"; break;
        case EXPECT_EOF: s = "end of file"; break;
        default:
            fprintf(stderr,"SHOULD NOT BE HERE!\nexpect = %i\n",expect);
            exit(1);
    }
    return s;
}


/**
 * Prints and error message and quits
 * @param msg The message to print (this argument may be NULL)
 * @note does not print line or position of error
 */
void parseErrMsg(const char *msg) {
    if (msg != NULL) {
        fprintf(stderr,"Error: %s\n", msg);
    }
    exit(1);
} 


/**
 * Prints a parse error message and terminates
 * @param msg The message to print (this argument may be NULL)
 * @param line the line in the input file where the error occurred
 * @param pos the position within that line where the error occurred 
 */
void parseErrXY(const char *msg, int64_t line, int64_t pos) {
    if (msg != NULL) {
        fprintf(stderr, "Error (line %" PRIu64 ", pos %" PRIu64 ") %s\n",
            line, pos, msg);
    }
    exit(1);
}

/**
 * Reports a width mismatch between two lines of a lex file.
 * @param tokenStartLine the input file line which produced the mismatch
 * @param sourceWidthLine the input file line which is used as a reference point
 * for the 'correct' width
 */
void parseErrWidthMismatch(uint64_t tokenStartLine, uint64_t sourceWidthLine) {
    fprintf(stderr, "Error (line %" PRIu64 ") Grid width mismatch with line " \
        "%" PRIu64 "\n", tokenStartLine, sourceWidthLine);
    exit(1);
}


/**
 * Prints a parse error message indicating the start of the current token and 
 * terminates
 * @param ps the current ParseState used by the parser
 * @param msg The message to print (this argument may be NULL)
 */
void parseErr(ParseState *ps, const char *msg) {
    parseErrXY(msg, ps->fileLineNo, ps->tokenStartPos);
}


/**
 * Prints a parse error message indicating an unexpected charater and terminates
 * @param ps the current ParseState used by the parser
 * @param ch the unexpected character
 */
void parseErrUnexpectedChar(ParseState *ps, char ch) {
    char buf[128];

    sprintf(buf, "Unexpected character \"%c\" (%02Xh) in input file", 
        (isprint(ch)?ch:' '), ch);
    parseErr(ps, buf);
}


/**
 * Prints a parse error message where an element of a pair was expected but
 * another token was found, then terminates
 * @param ps the current ParseState used by the parser
 * @param expToken the token which was expected
 * @param foundEof TRUE if end of file was encountered, FALSE otherwise
 * @param foundCh the character which was found instead of the expected token
 * @note if 'foundEof' is TRUE, then the 'foundCh' param is ignored
 */
void parseErrExpectedButFound(ParseState *ps,
        int expToken, bool foundEof, char foundCh) {
    char buf[128], buf2[64];
    const char *tail;

    if (foundEof) {
        tail = "end of file";
    } else {
        sprintf(buf2,"'%c' (%02Xh)", (isprint(foundCh)?foundCh:' '), foundCh);
        tail = buf2;
    }
    sprintf(buf, "In pair starting at (line %" PRIu64 ", pos %" PRIu64 "), " \
        "expected %s but found %s.\n",
        ps->pairStartLine, ps->pairStartPos, expectString(expToken), tail);
    
    if (foundEof) {
        fprintf(stderr,"%s", buf);
        exit(1);
    } else {
        parseErr(ps, buf);
    }
}


/**
 * Checks that the values in the ParserInfo struct are some-what valid
 * @param pi the struct to check
 * @note Does not return if pi contains invalid info
 */
void checkParserInfo( ParserInfo *pi) {
    if (pi->fileBufSize == 0) {
        parseErrMsg("Input file is empty");
    }
    if (pi->widthVal == NULL) {
        parseErrMsg("'width' is NULL");
    }
    if (pi->heightVal == NULL) {
        parseErrMsg("'height' is NULL");
    }
    if (pi->readCharFunc == NULL) {
        parseErrMsg("'readCharFunc' is NULL");
    }
    if (pi->getCharFunc == NULL) {
        parseErrMsg("'getCharFunc' is NULL");
    }
}


/**
 * Parses a lex or map file after guessing which kind it is
 * @param pi a structure containing info required by this function
 * @note This function guesses the file type by counting lex chars (supplied
 * by the caller) and '[' (which you can expect to see frequently in a 'map'
 * file).  If there are more 'lex' chars, it's a lex file, otherwise its a
 * map file.
 */
void parseByGuess(ParserInfo *pi) {
    uint64_t        pos =0;
    uint64_t        numLex=0, numMap=0;
    char            ch;
    bool            parseByLex = false;
    int             lexCharsAliveLen, lexCharsDeadLen;
    ParseState      ps;

    lexCharsAliveLen = strlen(pi->lexCharsAlive);
    lexCharsDeadLen = strlen(pi->lexCharsDead);
    ps.fileBufSize = pi->fileBufSize;

    while (pos < pi->fileBufSize) {
        ch = (*pi->getCharFunc)(&ps, pos++);
        if ((ch=='/') && ( (*pi->getCharFunc)(&ps, pos+1) == '/')) {
            do {
                pos++;
            } while ((pos < pi->fileBufSize) && 
                    ((*pi->getCharFunc)(&ps, pos) != CHAR_EOL));
            continue;
        }

        if (memchr(pi->lexCharsAlive,ch,lexCharsAliveLen) !=NULL) {
            numLex++;
        } else if (memchr(pi->lexCharsDead,ch,lexCharsDeadLen) !=NULL) {
            numLex++;
        } else if (ch=='[') {
            numMap++;
        }
    }
    parseByLex = numLex > numMap;

    if (pi->doVerbose) {
        printf("found %lu lex and %lu map\n",
            (unsigned long)numLex, (unsigned long)numMap);
        printf("Assuming %s input file\n",(parseByLex?"lex":"map"));
    }

    if (parseByLex) {
        parseLex(pi);
    } else {
        parseMap(pi);
    }
}


/**
 * Parses a lex file
 * @param pi a structure containing info required by this function
 * @note A lex file is to parse the map types seen at
 * https://conwaylife.com/ref/lexicon/lex_home.htm
 * Each map is described by a series of '.' (period, which indicates a dead
 * cell) and 'O' (capital letter 'O' which indicates a live cell).  Each row
 * ends with a line break.  So the maps should look something like
 * ..O..
 * .O.O.
 * O...O
 * @note Anything in the input file from a "//" to the end of line is a comment
 * and is ignored by the parser
 */
void parseLex(ParserInfo *pi) {
    char ch;
    int pass;
    uint64_t curLineGridWidth=0, widthSourceLine=0, curGridX=1, curGridY=1,
             numCells = 0;
    bool charAlive, charDead;
    ParseState ps;

    *pi->widthVal = 0;
    *pi->heightVal = 0;
    ps.fileBufSize = pi->fileBufSize;

    /* This parser goes through the input file twice.
     * The first pass is to determine the grid dimensions and if all of the 
     * input lines agree what that grid width is.
     * The second pass is to populate the grid.
     */
    for(pass=0; pass<2; pass++) {
        ps.fileBufPos = 0;
        ps.fileLineNo = 1;
        ps.fileLinePos = 1;

        while (ps.fileBufPos < ps.fileBufSize) {
            ps.tokenStartPos = ps.fileLinePos;
            ps.tokenStartLine = ps.fileLineNo;

            ch = (*pi->readCharFunc)(&ps);

            if (ch==0) {
                PRINT_PARSE_TOKEN("nothing");
                continue;
            }
            if ((ch == CHAR_EOL) || 
                ((ch=='/') && ((*pi->getCharFunc)(&ps, ps.fileBufPos)=='/'))) {

                if (pass == 0) {
                    if (curLineGridWidth > 0) {
                        if (*pi->widthVal == 0) {
                            // we have the width
                            *pi->widthVal = curLineGridWidth;
                            widthSourceLine = ps.tokenStartLine;
                        } else {
                            if (curLineGridWidth != *pi->widthVal) {
                                parseErrWidthMismatch(
                                    ps.tokenStartLine, widthSourceLine);
                            }
                        }
                        curLineGridWidth = 0;
                        (*pi->heightVal) ++;
                    }
                } else {
                    curGridY++;
                    curGridX = 1;
                }
                if (ch == CHAR_EOL) {
                    continue;
                }
            }

            // if comment...
            if ((ch == '/') && ((*pi->getCharFunc)(&ps, ps.fileBufPos)=='/')) {
                PRINT_PARSE_TOKEN("comment");
                // ... read to end of line
                do
                {
                    (*pi->readCharFunc)(&ps);
                } while((ps.fileLinePos>1) && (ps.fileBufPos < ps.fileBufSize));
                continue;
            }

            if (isspace(ch)) {
                PRINT_PARSE_TOKEN("whitespace");
                continue;
            }

            charAlive = strchr(pi->lexCharsAlive, ch);
            charDead = strchr(pi->lexCharsDead, ch);

            if (charAlive || charDead) {
                if (charAlive) {
                    PRINT_PARSE_TOKEN("live cell");
                } else {
                    PRINT_PARSE_TOKEN("dead cell");
                }
                if (pass == 0) {
                    curLineGridWidth++;
                    numCells++;
                } else {
                    (*pi->setCellFunc)(curGridX, curGridY,charAlive);
                    curGridX++;
                }
                continue;
            }

            /* Have checked for all valid inputs,
            *  so if we get here there's a problem.
            */
            parseErrUnexpectedChar(&ps, ch);
        }

        if (*pi->widthVal == 0) {
            fprintf(stderr,"No grid characters were found in the input file");
            exit(1);
        }
        if (pass == 0) {
            if ((numCells == 0) && pi->showWarns) {
                fprintf(stderr,"Warning: The input file does not specify any " \
                    "cells\n");
            }
            if (curLineGridWidth > 0) {

                if (curLineGridWidth != *pi->widthVal) {
                    parseErrWidthMismatch(ps.tokenStartLine, widthSourceLine);
                }
                (*pi->heightVal) ++;
            }

            (*pi->initGridFunc)();
        }
    }
}


/**
 * Parses a map file
 * @param pi a structure containing info required by this function
 * @note A map file should be a sequence of 'pairs', which look like "[x,y]".
 * The first pair in the file should describe the dimensions of the grid.  Every
 * subsequent pair should describe the location of a live cell.
 * @note Anything in the input file from a "//" to the end of line is a comment
 * and is ignored by the parser
 */
void parseMap(ParserInfo *pi) {
    char ch;
    bool parsingX=true, haveGridSize = false, nextIsDigit;
    int expect = EXPECT_NONE;
    uint64_t xval=0, yval=0, val;
    uint64_t numStartPos, numCells = 0;
    ParseState ps;

    ps.fileBufPos = 0;
    ps.fileLineNo = 1;
    ps.fileLinePos = 1;
    ps.fileBufSize = pi->fileBufSize;
    haveGridSize = false;

    while (ps.fileBufPos < ps.fileBufSize) {
        ps.tokenStartPos = ps.fileLinePos;
        ps.tokenStartLine = ps.fileLineNo;
        ch = (pi->readCharFunc)(&ps);

#if (PRINT_PARSE != 0)
        printf("\n[%c %02X] ",(isprint(ch)?ch:' '),ch);
#endif

        // nothing char
        if (ch == 0) {
            PRINT_PARSE_TOKEN("nothing");
            continue;
        }

        // skip whitespace
        if (isspace(ch)) {
            PRINT_PARSE_TOKEN("whitespace");
            continue;
        }

        // if comment...
        if ((ch == '/') && ((*pi->getCharFunc)(&ps, ps.fileBufPos)=='/')) {
            PRINT_PARSE_TOKEN("comment");
            // ... read to end of line
            do
            {
                (*pi->readCharFunc)(&ps);
            } while ((ps.fileLinePos>1) && (ps.fileBufPos < ps.fileBufSize));
            continue;
        }

        // start of a [x,y] pair
        if (ch == '[') {
            if (expect != EXPECT_NONE) {
                parseErr(&ps, "Unexpected char");
            }
            PRINT_PARSE_TOKEN("pair start");
            expect = EXPECT_NUMBER;
            ps.pairStartLine = ps.fileLineNo;
            ps.pairStartPos = ps.fileLinePos;
            continue;
        }
        
        if (isdigit(ch)) {
            if (expect != EXPECT_NUMBER) {
                parseErr(&ps, "Unexpected number");
            }
            if (PRINT_PARSE) printf("number");
            numStartPos = ps.fileBufPos-1;
            val = 0;
            do  {
                val = val * 10 + ch - '0';
                nextIsDigit = isdigit((*pi->getCharFunc)(&ps, ps.fileBufPos));
                if (nextIsDigit) {
                    // 17 is two less than the maximum number of digits in a
                    //  uint64_t.
                    if (ps.fileBufPos - numStartPos > 17) {
                        parseErrXY("Number is too long", ps.fileLineNo, 
                            numStartPos);
                    }
                    ch = (*pi->readCharFunc)(&ps);
                }
            }while (nextIsDigit);
            if (parsingX) {
                xval = val;
                expect = EXPECT_COMMA;
                parsingX = false;
                if (haveGridSize && (val > *pi->widthVal)) {
                    parseErr(&ps, "X value is greater that grid width");
                }
            } else {
                yval = val;
                parsingX = true;
                expect = EXPECT_CLOSEBR;
                if (haveGridSize && (val > *pi->heightVal)) {
                    parseErr(&ps, "Y value is greater than grid height");
                }
            }
            continue;
        }

        if (ch == ']') {
            if (expect != EXPECT_CLOSEBR) {
                parseErr(&ps, "Expected ']'");
            }
            PRINT_PARSE_TOKEN("']'");
            expect = EXPECT_NONE;
            if (haveGridSize) {
                if (xval >= *pi->widthVal) {
                    parseErr(&ps, "X value is outside the grid");
                }
                if (yval >= *pi->heightVal) {
                    parseErr(&ps, "Y value is outside the grid");
                }

                if (((*pi->getCellFunc)(xval+1,yval+1)==1) && pi->showWarns) {
                    fprintf(stderr,"Warning: There is a duplicate cell " \
                        "[%" PRIu64 ":%" PRIu64 "] specified in the pair " \
                        "beginning at line " \
                        " %" PRIu64 ", pos %" PRIu64 "\n",
                        xval, yval, ps.pairStartLine, ps.pairStartPos
                        );
                }

                (*pi->setCellFunc)(xval+1,yval+1,1);
                numCells++;
            } else {
                *pi->widthVal = xval;
                *pi->heightVal = yval;
                (*pi->initGridFunc)();
                haveGridSize = true;
            }
            continue;
        }

        if (ch == ',') {
            if (expect != EXPECT_COMMA) {
                parseErr(&ps, "Expected comma");
            }
            PRINT_PARSE_TOKEN("comma");
            expect = EXPECT_NUMBER;
            continue;
        }

        /* Have checked for all valid inputs,
        *  so if we get here there's a problem.
        */
        parseErrUnexpectedChar(&ps, ch);
    }

    if (expect != EXPECT_NONE) {
        parseErrExpectedButFound(&ps,expect,true,0);
    }
    if (!haveGridSize) {
        parseErr(&ps, "No grid size specified");
    }
    if ((numCells == 0) && pi->showWarns) {
        fprintf(stderr,"Warning: The input file does not specify any cells\n");
    }
}


