/*
 *  Written by: Brent Feulbach
 *  Date: 20-Aug-2020
 */

#ifndef INC_LIFEPARS_H


/* INCLUDES */

/*
 * These macros allow one to printf fixed size values.  See:
 * https://en.cppreference.com/w/cpp/types/integer
 */ 
#define __STDC_LIMIT_MACROS
#define __STDC_CONSTANT_MACROS
#define __STDC_FORMAT_MACROS

#include <assert.h>
#include <chrono>
#include <ctype.h>
#include <errno.h>
#include <inttypes.h>
#include <limits.h>
#include <math.h>
#include <stdbool.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <unistd.h>


/* DEFINITIONS */

#define PRINT_PARSE 0                   // print parsing info (for debugging)

#define CHAR_EOL    0x0A                // some ASCII characters of interest to
#define CHAR_CR     0x0D                //   the parser routines
#define CHAR_SPACE  0x20

// The following a used by the parsing code.  They describe the next token
// which is expected.
#define EXPECT_NONE     0               // nothing in partiular
#define EXPECT_NUMBER   1
#define EXPECT_COMMA    2
#define EXPECT_CLOSEBR  3
#define EXPECT_EOF      5               // end of file


/* MACROS */

// If print parse is not 0, the 'PRINT_PARSE_TOKEN' macro will print a message,
// otherwise it will do nothing.
#if (PRINT_PARSE != 0)
    #define PRINT_PARSE_TOKEN(s) printf("%s",s);
#else
    #define PRINT_PARSE_TOKEN(s) ((void)0)
#endif



/* TYPEDEFS */

struct tagParseState;

typedef unsigned char (*ReadFileCharFunc)(tagParseState *);
typedef unsigned char (*GetFileCharFunc)(tagParseState *, uint64_t);
typedef void (*SetCellFunc)(uint64_t,uint64_t,int);
typedef unsigned char (*GetCellFunc)(uint64_t,uint64_t);
typedef void (*InitGridFunc)(void);



/* STRUCTURES */

typedef struct tagParserInfo {
    uint64_t            fileBufSize;
    uint64_t            *widthVal;
    uint64_t            *heightVal;
    ReadFileCharFunc    readCharFunc;
    GetFileCharFunc     getCharFunc;
    const char          *lexCharsAlive;
    const char          *lexCharsDead;
    bool                showWarns;
    bool                doVerbose;
    GetCellFunc         getCellFunc;
    SetCellFunc         setCellFunc;
    InitGridFunc        initGridFunc;
} ParserInfo;


typedef struct tagParseState {
    uint64_t        fileBufSize;
    uint64_t        fileBufPos;
    uint64_t        fileLineNo;
    uint64_t        fileLinePos;
    uint64_t        tokenStartLine;
    uint64_t        tokenStartPos;
    uint64_t        pairStartLine;
    uint64_t        pairStartPos;
} ParseState;



/* FUNCTION PROTOTYPES */

#ifdef __cplusplus
extern "C" {
#endif

void parseByGuess(ParserInfo*);
void parseCmdLineDef(int, char**);
void parseCmdLine(int, char **);
bool parseDouble(const char *, double *);
void parseLex(ParserInfo*);
void parseMap(ParserInfo*);
bool parseUint64(const char *, uint64_t *);
bool parseUHex64(const char *, uint64_t *);


#ifdef __cplusplus
}
#endif


#define INC_LIFEPARS_H
#endif